module rayd.scene;

private {
    import std.file;
    import std.string;
    import std.json;
    import gl3n.linalg;
    import gl3n.math : PI, sqrt, tan;
    import rayd.tracer;
}


class Camera
{
    const @property vec3 Position()
    {
        return m_Position;
    }
    @property void Position(vec3 value)
    {
        m_Position = value;
    }

    const @property vec3 Direction()
    {
        return m_Direction;
    }
    @property void Direction(vec3 value)
    {
        m_Direction = value;
    }

    const @property vec3 UpVector()
    {
        return m_UpVector;
    }
    @property void UpVector(vec3 value)
    {
        m_UpVector = value;
    }

    const @property float FOV()
    {
        return m_FOV;
    }
    @property void FOV(float value)
    {
        m_FOV = value;
    }

    const @property vec3 U()
    {
        return m_U;
    }

    const @property vec3 V()
    {
        return m_V;
    }

    void Setup(float aspectRatio)
    {
        vec3 u = cross(this.UpVector, this.Direction);
        vec3 v = cross(this.Direction, u);

        float planeWidth = tan((this.FOV * (PI / 180.0f)) * 0.5f);

        float invULen = 1.0f / u.length;
        float invVLen = 1.0f / v.length;

        m_U = u * (planeWidth * aspectRatio) * invULen;
        m_V = v * planeWidth * invVLen;
    }

    Ray ComputeRay(float x, float y, float width, float height) const
    {
        vec3 rayDir = this.Direction + (this.U * ((x / (width - 1.0f)) * 2.0f - 1.0f)) - (this.V * ((y / (height - 1.0f)) * 2.0f - 1.0f));
        return new Ray(this.Position, rayDir);
    }


private {
    vec3            m_Position;
    vec3            m_Direction;
    vec3            m_UpVector;
    float           m_FOV;

    // screen interpolation
    vec3            m_U;
    vec3            m_V;
}
}



enum PrimitiveType {
    Sphere = 0,
    Plane,
    Light
}


class Primitive
{
    int Intersection(in Ray ray, ref float distance, ref RayHitInfo hitInfo)
    {
        return 0;
    }



    const @property PrimitiveType Type()
    {
        return m_Type;
    }

    const @property vec3 Position()
    {
        return m_Position;
    }
    @property void Position(vec3 value)
    {
        m_Position = value;
    }

    const @property vec3 Color()
    {
        return m_Color;
    }
    @property void Color(vec3 value)
    {
        m_Color = value;
    }

    const @property float Specular()
    {
        return m_Specular;
    }
    @property void Specular(float value)
    {
        m_Specular = value;
    }

private {
    PrimitiveType   m_Type;
    vec3            m_Position;
    vec3            m_Color;
    float           m_Specular;
}
}

class Plane : Primitive
{
    this()
    {
        m_Type = PrimitiveType.Plane;
    }


    override int Intersection(in Ray ray, ref float distance, ref RayHitInfo hitInfo)
    {
        float d = dot(this.Normal, ray.Direction);
        if (d != 0)
        {
            float dist = -(dot(this.Normal, ray.Direction) + this.Distance) / d;
            if (dist > 0.0f)
            {
                if (dist < distance) 
                {
                    distance = dist;
                    hitInfo.hitDistance = dist;
                    hitInfo.hitPosition = ray.Origin + (ray.Direction * dist);
                    hitInfo.hitNormal = this.Normal;
                    hitInfo.hitColor = this.Color;
                    hitInfo.hitSpecular = this.Specular;
                    return 1;
                }
            }
        }
        return 0;
    }


    const @property vec3 Normal()
    {
        return m_Normal;
    }
    @property void Normal(vec3 value)
    {
        m_Normal = value;
    }

    const @property float Distance()
    {
        return m_Distance;
    }
    @property void Distance(float value)
    {
        m_Distance = value;
    }


private {
    vec3    m_Normal;
    float   m_Distance;
}
}

class Sphere : Primitive
{
    this()
    {
        m_Type = PrimitiveType.Sphere;
    }


    override int Intersection(in Ray ray, ref float distance, ref RayHitInfo hitInfo)
    {
        vec3 v = ray.Origin - this.Position;
        float b = -dot(v, ray.Direction);
        float det = (b * b) - dot(v, v) + (this.Radius * this.Radius);
        int retval = 0;
        if (det > 0)
        {
            det = sqrt(det);
            float i1 = b - det;
            float i2 = b + det;
            if (i2 > 0)
            {
                if (i1 < 0)
                {
                    if (i2 < distance) 
                    {
                        distance = i2;
                        retval = 2;
                    }
                }
                else
                {
                    if (i1 < distance)
                    {
                        distance = i1;
                        retval = 1;
                    }
                }
            }
        }

        if (retval)
        {
            hitInfo.hitDistance = distance;
            hitInfo.hitPosition = ray.Origin + (ray.Direction * distance);
            hitInfo.hitNormal = (hitInfo.hitPosition - this.Position).normalized;
            hitInfo.hitColor = this.Color;
            hitInfo.hitSpecular = this.Specular;
        }

        return retval;
    }


    const @property float Radius()
    {
        return m_Radius;
    }
    @property void Radius(float value)
    {
        m_Radius = value;
    }

private {
    float   m_Radius;
}
}

class Light : Sphere
{
    this()
    {
        m_Type = PrimitiveType.Light;
    }
}




class Scene
{
    @property Primitive[] Primitives()
    {
        return m_Primitives;
    }

    @property Light[] Lights()
    {
        return m_Lights;
    }

    @property Camera MainCamera()
    {
        return m_Camera;
    }

    bool Load(string path)
    {
        bool result = false;

        try {
            string jsonStr = readText(path);
            JSONValue json = parseJSON(jsonStr);
            if (json.type != JSON_TYPE.NULL)
            {
                // loading primitives
                JSONValue primitives = json["primitives"];
                m_Primitives = new Primitive[primitives.object.length];
                int i = 0;
                foreach (name; primitives.object.keys)
                {
                    if (name.startsWith("plane"))
                    {
                        m_Primitives[i] = this.LoadPlane(primitives[name]);
                    }
                    else if (name.startsWith("sphere"))
                    {
                        m_Primitives[i] = this.LoadSphere(primitives[name]);
                    }
                    else
                        assert(false, "Unknown primitive type!");

                    ++i;
                }

                // loading lights
                i = 0;
                JSONValue lights = json["lights"];
                m_Lights = new Light[lights.object.length];
                foreach (name; lights.object.keys)
                {
                    m_Lights[i] = this.LoadLight(lights[name]);
                    ++i;
                }

                // loading camera
                JSONValue camera = json["camera"];
                this.LoadCamera(camera);

                result = true;
            }
        } catch {
        }

        return result;
    }


private {
    // iOrange - little function that helps to get float value
    real GetJSONFloat(JSONValue v)
    {
        real ret = 0.0f;
        if (v.type == JSON_TYPE.FLOAT)
            ret = v.floating;
        else if (v.type == JSON_TYPE.INTEGER)
            ret = cast(real)v.integer;
        else if (v.type == JSON_TYPE.UINTEGER)
            ret = cast(real)v.uinteger;

        return ret;
    }

    vec3 GetJSONVec3(JSONValue obj, string name)
    {
        return vec3(GetJSONFloat(obj[name][0]),
                    GetJSONFloat(obj[name][1]),
                    GetJSONFloat(obj[name][2]));
    }

    Plane LoadPlane(JSONValue json)
    {
        Plane p = new Plane();
        p.Normal = GetJSONVec3(json, "normal");
        p.Distance = GetJSONFloat(json["distance"]);
        p.Specular = GetJSONFloat(json["specular"]);
        p.Color = GetJSONVec3(json, "color");

        return p;
    }

    Sphere LoadSphere(JSONValue json)
    {
        Sphere s = new Sphere();
        s.Position = GetJSONVec3(json, "pos");
        s.Radius = GetJSONFloat(json["radius"]);
        s.Specular = GetJSONFloat(json["specular"]);
        s.Color = GetJSONVec3(json, "color");

        return s;
    }

    Light LoadLight(JSONValue json)
    {
        Light l = new Light();
        l.Position = GetJSONVec3(json, "pos");
        l.Color = GetJSONVec3(json, "color");

        return l;
    }

    void LoadCamera(JSONValue json)
    {
        m_Camera = new Camera();
        m_Camera.Position = GetJSONVec3(json, "pos");
        m_Camera.Direction = GetJSONVec3(json, "dir");
        m_Camera.UpVector = GetJSONVec3(json, "up");
        m_Camera.FOV = GetJSONFloat(json["fov"]);
    }
}


private {
    Primitive[]     m_Primitives;
    Light[]         m_Lights;
    Camera          m_Camera;
}
}
