module rayd.tracer;

private {
    import gl3n.linalg;
    import rayd.scene;
    import std.math : pow;
}


private vec3 reflect(vec3 dir, vec3 normal)
{
    vec3 temp = normal * dot(dir, normal) * 2.0f;
    return dir - temp;
}



struct RayHitInfo
{
    float   hitDistance;
    vec3    hitPosition;
    vec3    hitNormal;
    vec3    hitColor;
    float   hitSpecular;
};


class Ray
{
    this(vec3 origin, vec3 direction)
    {
        m_Origin = origin;
        m_Direction = direction.normalized;
    }

    const @property vec3 Origin()
    {
        return m_Origin;
    }
    const @property vec3 Direction()
    {
        return m_Direction;
    }

private {
    vec3    m_Origin;
    vec3    m_Direction;
}
}


class Tracer
{
    this()
    {
    }
    ~this()
    {
    }

    void Render(Scene scene, uint* image, int width, int height)
    {
        scene.MainCamera.Setup(cast(float)width / cast(float)height);

        for (int y = 0; y < height; ++y)
        {
            for (int x = 0; x < width; ++x)
            {
                Ray ray = scene.MainCamera.ComputeRay(cast(float)x, cast(float)y, cast(float)width, cast(float)height);

                vec3 color = this.Trace(scene, ray);
                int r = cast(int)(color.r * 255.0f);
                int g = cast(int)(color.g * 255.0f);
                int b = cast(int)(color.b * 255.0f);

                image[x + y * width] = 0xff000000 | (b << 16) | (g << 8) | r;
            }
        }
    }


private {
    bool FindIntersection(Scene scene, Ray ray, ref RayHitInfo hitInfo, ref float distance)
    {
        bool found = false;

        foreach (prim; scene.Primitives)
        {
            if (prim.Intersection(ray, distance, hitInfo))
            {
                found = true;
            }
        }

        return found;
    }

    vec3 Trace(Scene scene, Ray ray)
    {
        vec3 result = vec3(0.0f, 0.0f, 0.0f);

        float distance = 1000000.0f;
        RayHitInfo hitInfo;
        if (this.FindIntersection(scene, ray, hitInfo, distance))
        {
            result = this.Shader(ray, hitInfo, scene);
        }

        return result;
    }

    vec3 Shader(Ray ray, RayHitInfo hitInfo, Scene scene)
    {
        vec3 color = vec3(0.0f, 0.0f, 0.0f);

        vec3 matDiffuse = hitInfo.hitColor;

        foreach (light; scene.Lights)
        {
            vec3 dirToLight = light.Position - hitInfo.hitPosition;
            float distToLight = dirToLight.length;
            dirToLight.normalize();
            float NdotL = dot(hitInfo.hitNormal, dirToLight);

            Ray lightRay = new Ray(hitInfo.hitPosition + (hitInfo.hitNormal * 0.001f), dirToLight);
            RayHitInfo lightHitInfo;
            if (!this.FindIntersection(scene, lightRay, lightHitInfo, distToLight))
            {
                float lambert = clamp(NdotL, 0.0f, 1.0f);
                if (lambert > 0.0f)
                {
                    color.x += light.Color.x * matDiffuse.x * lambert;
                    color.y += light.Color.y * matDiffuse.y * lambert;
                    color.z += light.Color.z * matDiffuse.z * lambert;

                    // point light source: sample once for specular highlight
                    if (hitInfo.hitSpecular > 0.0f)
                    {
                        vec3 V = ray.Direction;
                        vec3 R = reflect(dirToLight, hitInfo.hitNormal).normalized;
                        float VdotR = dot(V, R);
                        if (VdotR > 0.0f)
                        {
                            float spec = pow(VdotR, 32.0f) * lambert * hitInfo.hitSpecular;
                            // add specular component to ray color
                            color.x += spec * light.Color.x;
                            color.y += spec * light.Color.y;
                            color.z += spec * light.Color.z;
                        }
                    }
                }
            }
        }

        color.x = clamp(color.x, 0.0f, 1.0f);
        color.y = clamp(color.y, 0.0f, 1.0f);
        color.z = clamp(color.z, 0.0f, 1.0f);

        return color;
    }
}
}
