module rayd.scene;

private {
    import std.file;
    import std.string;
    import std.json;
    import gl3n.linalg;
    import gl3n.math : sqrt;
    import rayd.tracer;
}



enum PrimitiveType {
    Sphere = 0,
    Plane,
    Light
}


class Primitive
{
    abstract int Intersection(in Ray ray, ref float distance);

    @property const PrimitiveType Type()
    {
        return m_Type;
    }

    @property const vec3 Position()
    {
        return m_Position;
    }
    @property void Position(vec3 value)
    {
        m_Position = value;
    }

    @property const vec3 Color()
    {
        return m_Color;
    }
    @property void Color(vec3 value)
    {
        m_Color = value;
    }

private {
    PrimitiveType   m_Type;
    vec3            m_Position;
    vec3            m_Color;
}
}

class Plane : Primitive
{
    this()
    {
        m_Type = PrimitiveType.Plane;
    }


    override int Intersection(in Ray ray, ref float distance)
    {
        float d = dot(this.Normal, ray.Direction);
        if (d != 0)
        {
            float dist = -(dot(this.Normal, ray.Direction) + this.Distance) / d;
            if (dist > 0.0f)
            {
                if (dist < distance)
                {
                    distance = dist;
                    return 1;
                }
            }
        }
        return 0;
    }


    @property const vec3 Normal()
    {
        return m_Normal;
    }
    @property void Normal(vec3 value)
    {
        m_Normal = value;
    }

    @property const float Distance()
    {
        return m_Distance;
    }
    @property void Distance(float value)
    {
        m_Distance = value;
    }


private {
    vec3    m_Normal;
    float   m_Distance;
}
}

class Sphere : Primitive
{
    this()
    {
        m_Type = PrimitiveType.Sphere;
    }


    override int Intersection(in Ray ray, ref float distance)
    {
        vec3 v = ray.Origin - this.Position;
        float b = -dot(v, ray.Direction);
        float det = (b * b) - dot(v, v) + (this.Radius * this.Radius);
        int retval = 0;
        if (det > 0)
        {
            det = sqrt(det);
            float i1 = b - det;
            float i2 = b + det;
            if (i2 > 0)
            {
                if (i1 < 0)
                {
                    if (i2 < distance)
                    {
                        distance = i2;
                        retval = 2;
                    }
                }
                else
                {
                    if (i1 < distance)
                    {
                        distance = i1;
                        retval = 1;
                    }
                }
            }
        }
        return retval;
    }


    @property const float Radius()
    {
        return m_Radius;
    }
    @property void Radius(float value)
    {
        m_Radius = value;
    }

private {
    float   m_Radius;
}
}




class Scene
{
    @property Primitive[] Primitives()
    {
        return m_Primitives;
    }

    bool Load(string path)
    {
        bool result = false;

        try {
            string jsonStr = readText(path);
            JSONValue json = parseJSON(jsonStr);
            if (json.type != JSON_TYPE.NULL)
            {
                // loading primitives
                JSONValue primitives = json["primitives"];
                m_Primitives = new Primitive[primitives.object.length];
                int i = 0;
                foreach (name; primitives.object.keys)
                {
                    if (name.startsWith("plane"))
                    {
                        m_Primitives[i] = this.LoadPlane(primitives[name]);
                    }
                    else if (name.startsWith("sphere"))
                    {
                        m_Primitives[i] = this.LoadSphere(primitives[name]);
                    }
                    else
                        assert(false, "Unknown primitive type!");

                    ++i;
                }

                result = true;
            }
        } catch {
        }

        return result;
    }


private {
    // iOrange - little function that helps us to get float value
    real GetJSONFloat(JSONValue v)
    {
        real ret = 0.0f;
        if (v.type == JSON_TYPE.FLOAT)
            ret = v.floating;
        else if (v.type == JSON_TYPE.INTEGER)
            ret = cast(real)v.integer;
        else if (v.type == JSON_TYPE.UINTEGER)
            ret = cast(real)v.uinteger;

        return ret;
    }

    vec3 GetJSONVec3(JSONValue obj, string name)
    {
        return vec3(GetJSONFloat(obj[name][0]),
                    GetJSONFloat(obj[name][1]),
                    GetJSONFloat(obj[name][2]));
    }

    Plane LoadPlane(JSONValue json)
    {
        Plane p = new Plane();
        p.Normal = GetJSONVec3(json, "normal");
        p.Distance = GetJSONFloat(json["distance"]);
        p.Color = GetJSONVec3(json, "color");

        return p;
    }

    Sphere LoadSphere(JSONValue json)
    {
        Sphere s = new Sphere();
        s.Position = GetJSONVec3(json, "pos");
        s.Radius = GetJSONFloat(json["radius"]);
        s.Color = GetJSONVec3(json, "color");

        return s;
    }
}


private {
    Primitive[]     m_Primitives;
}
}
