#include <stdafx.h>
#include "TheoraPlayer.h"


TheoraPlayer::TheoraPlayer()
	: m_VideoOpened(false)
	, m_File(0)
	, m_VideoTimer(0)
	, m_LastVideoFrame(~0)
	, m_FrameWidth(0)
	, m_FrameHeight(0)
	, m_FramesPerSecond(0)
	, m_Stopped(true)
{
}
TheoraPlayer::~TheoraPlayer()
{
	if (m_File)
	{
		fclose(m_File);
		m_File = 0;
	}
}


// iOrange - opens video from file using "fileName", return true if all is ok, false if not
bool TheoraPlayer::OpenVideoFile(const std::string& fileName)
{
	m_VideoOpened = false;

	// iOrange - init structs
	ogg_sync_init(&m_OggSyncState);
	theora_comment_init(&m_TheoraComment);
	theora_info_init(&m_TheoraInfo);

	if (this->OpenFile(fileName))
	{
		bool dataFound = false;
		int theoraPacketsFound = 0;

		while (!dataFound)
		{
			// iOrange - grab some data from file and put it into the ogg stream
			this->BufferData();

			// iOrange - grab the ogg page from the stream
			while (ogg_sync_pageout(&m_OggSyncState, &m_OggPage) > 0)
			{
				ogg_stream_state test;

				// iOrange - check: if this is not headers page, then we finished
				if (!ogg_page_bos(&m_OggPage))
				{
					// iOrange - all headers pages are finished, now there are only data packets
					dataFound = true;

					// iOrange - don't leak the page, get it into the video stream
					ogg_stream_pagein(&m_VideoStream, &m_OggPage);
					break;
				}

				// iOrange - we nee to identify the stream

				// 1) Init the test stream with the s/n from our page
				ogg_stream_init(&test, ogg_page_serialno(&m_OggPage));
				// 2) Add this page to this test stream
				ogg_stream_pagein(&test, &m_OggPage);
				// 3) Decode the page into the packet
				ogg_stream_packetout(&test, &m_OggPacket);

				// iOrange - try to interpret the packet as Theora's data
				if(!theoraPacketsFound && theora_decode_header(&m_TheoraInfo, &m_TheoraComment, &m_OggPacket) >= 0)
				{
					// iOrange - theora found ! Let's copy the stream
					memcpy(&m_VideoStream, &test, sizeof(test));
					theoraPacketsFound++;
				}
				else
				{
					// iOrange - non-theora (vorbis maybe)
					ogg_stream_clear(&test);
				}
			}
		}

		// iOrange - no theora found, maybe this is music file ?
		if (theoraPacketsFound)
		{
			int err;
			// iOrange - by specification we need 3 header packets for any logical stream (theora, vorbis, etc.)
			while (theoraPacketsFound < 3)
			{
				err = ogg_stream_packetout(&m_VideoStream, &m_OggPacket);
				if (err < 0)
				{
					// iOrange - some stream errors (maybe stream corrupted?)
					break;
				}
				if (err > 0)
				{
					if (theora_decode_header(&m_TheoraInfo, &m_TheoraComment, &m_OggPacket) >= 0)
						theoraPacketsFound++;
					else
					{
						// iOrange - another stream corruption ?
						break;
					}
				}

				// iOrange - if read nothing from packet - just grab more data into packet
				if (!err)
				{
					if (ogg_sync_pageout(&m_OggSyncState, &m_OggPage) > 0)
					{
						// iOrange - if data arrivet into packet - put it into our logical stream
						ogg_stream_pagein(&m_VideoStream, &m_OggPage);
					}
					else
					{
						// iOrange - nothing goint from the ogg stream, need to read some data from file
						if (!this->BufferData())
						{
							// iOrange - f***k ! End of file :(
							break;
						}
					}
				}
			}
		}

		// iOrange - if we have theora ok
		if (theoraPacketsFound)
		{
			// iOrange - init decoder
			if (0 == theora_decode_init(&m_TheoraState, &m_TheoraInfo))
			{
				// iOrange - decoder intialization succeed
				m_VideoOpened = true;

				m_FrameWidth = m_TheoraInfo.frame_width;
				m_FrameHeight = m_TheoraInfo.frame_height;
				m_FramesPerSecond = static_cast<float>(m_TheoraInfo.fps_numerator) / static_cast<float>(m_TheoraInfo.fps_denominator);
				m_VideoTimer = 0;
				m_Stopped = false;
			}
		}
	}

	return m_VideoOpened;
}

// iOrange - return video frame width
int TheoraPlayer::GetFrameWidth(void) const
{
	return m_FrameWidth;
}

// iOrange - return video frame height
int TheoraPlayer::GetFrameHeight(void) const
{
	return m_FrameHeight;
}

// iOrange - return video frequency in frames-per-second (FPS)
float TheoraPlayer::GetFramesPerSecond(void) const
{
	return m_FramesPerSecond;
}

// iOrange - advance video playing by "dt" milliseconds
uint32 TheoraPlayer::Advance(uint32 dt)
{
	if (m_Stopped)
		return m_LastVideoFrame;

	// iOrange - advance video timer by delta time in milliseconds (dt)
	m_VideoTimer += dt;

	// iOrange - calculate current frame
	const uint32 curFrame = static_cast<uint32>(floor(static_cast<float>(m_VideoTimer) * m_FramesPerSecond * 0.001f));
	if (m_LastVideoFrame != curFrame)
	{
		m_LastVideoFrame = curFrame;
		this->DecodeVideoFrame();
	}

	return curFrame;
}

// iOrange - return true if video is not playing
bool TheoraPlayer::IsStopped(void) const
{
	return m_Stopped;
}

// iOrange: clips t to range [0; 255]
inline int clip(int t)
{
	return ((t < 0) ? (0) : ((t > 255) ? 255 : t));
}

// iOrange - extract current frame to RGB color space using CPU for yuv-rgb conversion (integer optimization)
void TheoraPlayer::GetFrameRGB(char* outFrame, int pitch)
{
	for (int y = 0; y < m_FrameHeight; ++y)
	{
		for (int x = 0; x < m_FrameWidth; ++x)
		{
			const int off = x + y * pitch;
			const int xx = x >> 1;
			const int yy = y >> 1;

			const int Y = static_cast<int>(m_YUVFrame.y[x  +  y * m_YUVFrame. y_stride]) - 16;
			const int U = static_cast<int>(m_YUVFrame.u[xx + yy * m_YUVFrame.uv_stride]) - 128;
			const int V = static_cast<int>(m_YUVFrame.v[xx + yy * m_YUVFrame.uv_stride]) - 128;

			outFrame[off * 3 + 0] = clip((298 * Y           + 409 * V + 128) >> 8);
			outFrame[off * 3 + 1] = clip((298 * Y - 100 * U - 208 * V + 128) >> 8);
			outFrame[off * 3 + 2] = clip((298 * Y + 516 * U           + 128) >> 8);
		}
	}
}

// iOrange - extract current frame to planar YUV444 (de-interlaced yuv) to perform yuv-rgb conversion on GPU
void TheoraPlayer::GetFrameYUV444(char* outFrame, int pitch)
{
	for (int y = 0; y < m_FrameHeight; ++y)
	{
		for (int x = 0; x < m_FrameWidth; ++x)
		{
			const int off = x + y * pitch;
			const int xx = x >> 1;
			const int yy = y >> 1;

			outFrame[off * 3 + 0] = m_YUVFrame.y[x  +  y * m_YUVFrame. y_stride];
			outFrame[off * 3 + 1] = m_YUVFrame.u[xx + yy * m_YUVFrame.uv_stride];
			outFrame[off * 3 + 2] = m_YUVFrame.v[xx + yy * m_YUVFrame.uv_stride];
		}
	}
}


// iOrange - internal open file routines
bool TheoraPlayer::OpenFile(const std::string& fileName)
{
#if _MSC_VER >= 1300
	m_File = 0;
	if (0 != fopen_s(&m_File, fileName.c_str(), "rb"))
		return false;
#else
	m_File = fopen(fileName.c_str(), "rb");
	if (!m_File)
		return false;
#endif

	return true;
}

// iOrange - buffer some data from file into Ogg stream
int TheoraPlayer::BufferData(void)
{
	static const int k_SyncBufferSize = 4096;

	// iOrange - ask some buffer for putting data into stream
	char* buffer = ogg_sync_buffer(&m_OggSyncState, k_SyncBufferSize);
	// iOrange - read data from file
	int bytes = fread(buffer, 1, k_SyncBufferSize, m_File);
	// iOrange - put readed data into Ogg stream
	ogg_sync_wrote(&m_OggSyncState, bytes);
	return bytes;
}

// iOrange - decodes current frame from Theora to YUV
void TheoraPlayer::DecodeVideoFrame(void)
{
	// iOrange - first of all - grab some data into ogg packet
	while (ogg_stream_packetout(&m_VideoStream, &m_OggPacket) <= 0)
	{
		// iOrange - if no data in video stream, grab some data
		if (!this->BufferData())
		{
			m_Stopped = true;
			return;
		}

		// iOrange - grab all decoded ogg pages into our video stream
		while (ogg_sync_pageout(&m_OggSyncState, &m_OggPage) > 0)
			ogg_stream_pagein(&m_VideoStream, &m_OggPage);
	}

	// iOrange - load packet into theora decoder
	if (0 == theora_decode_packetin(&m_TheoraState, &m_OggPacket))
	{
		// iOrange - if decoded ok - get YUV frame
		theora_decode_YUVout(&m_TheoraState, &m_YUVFrame);
	}
	else
		m_Stopped = true;
}

