#include "MetroModelReader.h"

#include <shlwapi.h>
#pragma comment (lib, "shlwapi.lib")


#define CHUNK_GUARD_BEGIN	const size_t offset = ftell(file);
#define CHUNK_GUARD_END		fseek(file, offset + chunkSize, SEEK_SET);


MObject FindMObject(const MString& name)
{
	MSelectionList slist;
	slist.add( name );
	if (slist.length() == 0)
	{
		DebugPrint(name + MString(" not found (0)"));
		return MObject::kNullObj;
	}

	MObject obj;
	if (!slist.getDependNode(0, obj))
	{
		DebugPrint(name + MString(" not found (1)"));
		return MObject::kNullObj;
	}

	return obj;
}

/// retrieve the value of a given maya node "int" attribute
template<class T>
static bool GetAttribute(MFnDependencyNode& node, const char* name, T& val)
{
	MStatus status;
	MPlug p = node.findPlug(name, &status);
	if (MS::kSuccess != status)
	{
		return false;
	}

	status = p.getValue(val);
	return static_cast<bool>(status);
}

/// retrieve the value of a given maya node "int" attribute
template<class T>
static bool SetAttribute(MFnDependencyNode& node, const char* name, const T& val)
{
	MStatus status;
	MPlug p = node.findPlug(name, &status);
	if (MS::kSuccess != status)
	{
		DebugPrint(MString(name) + MString(" not found"));
		return false;
	}

	status = p.setValue(val);
	return static_cast<bool>(status);
}

void ConnectAttr(MDGModifier & mod, const MFnDependencyNode fnsrc, const MString & srcAttr,
									const MFnDependencyNode fndst, const MString & dstAttr)
{
	MStatus status;
	MPlug psrc = fnsrc.findPlug(srcAttr, &status);
	if (MS::kSuccess != status)
	{
		DebugPrint(srcAttr + MString(" not found (0)"));
		return;
	}

	MPlug pdst = fndst.findPlug(dstAttr, &status);
	if (MS::kSuccess != status)
	{
		DebugPrint(dstAttr + MString(" not found (1)"));
		return;
	}
	
	mod.connect(psrc, pdst);
}

void AttachShadingGroup(const MObject& sgroup, const MObject& gprim)
{
	MStatus status;

	MFnDependencyNode fnsg(sgroup, &status);
	if (MS::kSuccess != status)
	{
		DebugPrint(MString("sgroup") + MString(" not found (0)"));
		return;
	}

	MFnDagNode fngp(gprim, &status);
	if (MS::kSuccess != status)
	{
		DebugPrint(MString("gprim") + MString(" not found (1)"));
		return;
	}

	MString cmd = "sets -e -forceElement " + fnsg.name() + " " + fngp.fullPathName() + ";", result;
	MGlobal::executeCommand(cmd, result, false, false);
}




CMetroModelReader::CMetroModelReader()
	: m_CurMesh(NULL)
	, m_LastMaterials(NULL)
	, m_LastMaterialsSize(0)
{
	m_InitialShadingGroup = FindMObject("initialShadingGroup");
}
CMetroModelReader::~CMetroModelReader()
{
	for (meshesVec::iterator it = m_Meshes.begin(); it != m_Meshes.end(); ++it)
	{
		SMesh* mesh = *it;
		delete [] mesh->VBPtr;
		delete [] mesh->IBPtr;
		delete mesh;
	}
}

int CMetroModelReader::OpenModelFile(const std::string& fileName)
{
	FILE* file = NULL;
	fopen_s(&file, fileName.c_str(), "rb");
	if (!file)
	{
		DebugPrint(MString("Could not open ") + MString(fileName.c_str()));
		return FALSE;
	}

	fseek(file, 0, SEEK_END);
	const size_t fileSize = ftell(file);
	fseek(file, 0, SEEK_SET);

	size_t filePos = 0;
	while (filePos < fileSize)
	{
		unsigned int chunkID, chunkSize;
		fread(&chunkID, 4, 1, file);
		fread(&chunkSize, 4, 1, file);

		this->ReadChunk(chunkID, chunkSize, file);

		filePos += chunkSize + 8;
	}

	fclose(file);

	m_ContentPath = fileName;
	std::replace(m_ContentPath.begin(), m_ContentPath.end(), '/', '\\');
	std::string::size_type pos = m_ContentPath.rfind("\\content\\");
	if (pos != std::string::npos)
		m_ContentPath = m_ContentPath.substr(0, pos + 9);
	else
		m_ContentPath.erase();

	DebugPrint(MString("File opened OK ") + MString(fileName.c_str()));
	return TRUE;
}

int CMetroModelReader::ImportToMaya(MObject parent)
{
	if (m_Meshes.empty())
	{
		DebugPrint(MString("No meshes found"));
		return FALSE;
	}

	MString debStr = "Found ";
	debStr += static_cast<int>(m_Meshes.size());
	debStr += " meshes";
	DebugPrint(debStr);

	int i = 0;

	for (meshesVec::const_iterator it = m_Meshes.begin(); it != m_Meshes.end(); ++it, ++i)
	{
		debStr = "Constructing ";
		debStr += i;
		debStr += " mesh";
		DebugPrint(debStr);

		this->MakeMayaMesh(*it, i, parent);
	}

	return TRUE;
}

void CMetroModelReader::ReadChunk(unsigned int chunkID, unsigned int chunkSize, FILE* file)
{
	CHUNK_GUARD_BEGIN

	switch (chunkID)
	{
		case METRO_CHUNK_MESHESHEAP:
		{
			unsigned int chunkPos = 0;
			while (chunkPos < chunkSize)
			{
				unsigned int meshIdx, meshSize;
				fread(&meshIdx, 4, 1, file);
				fread(&meshSize, 4, 1, file);

				unsigned int internalChunkID, internalChunkSize;
				unsigned int curPos = 0;
				while (curPos < meshSize)
				{
					fread(&internalChunkID, 4, 1, file);
					fread(&internalChunkSize, 4, 1, file);

					this->ReadMeshesHeapChunk(internalChunkID, internalChunkSize, file);

					curPos += internalChunkSize + 8;
				}

				chunkPos += meshSize + 8;
			}
		} break;
	}

	CHUNK_GUARD_END
}

void CMetroModelReader::ReadMeshesHeapChunk(unsigned int chunkID, unsigned int chunkSize, FILE* file)
{
	CHUNK_GUARD_BEGIN

	switch (chunkID)
	{
		case METRO_CHUNK_UNKNOWN_1:	// iOrange - for now just skip it
			break;

		case METRO_CHUNK_MATERIALS:
		{
			m_LastMaterials = new char[chunkSize];
			fread(m_LastMaterials, chunkSize, 1, file);

			m_LastMaterialsSize = chunkSize;
		} break;

		case METRO_CHUNK_VERTEXDATA:
		{
			this->ReadVertexDataChunk(chunkSize, file);
		} break;

		case METRO_CHUNK_INDEXDATA:
		{
			this->ReadIndexDataChunk(chunkSize, file);
		} break;
	}

	CHUNK_GUARD_END
}

void CMetroModelReader::ReadVertexDataChunk(unsigned int chunkSize, FILE* file)
{
	assert(NULL == m_CurMesh);

	unsigned int vertexType = 0;

	SMesh* mesh = new SMesh;

	fread(&vertexType, 4, 1, file);
	fread(&mesh->numVerteces, 4, 1, file);

	mesh->VBPtr = new MetroModelVertex[mesh->numVerteces];
	fread(mesh->VBPtr, mesh->numVerteces * sizeof(MetroModelVertex), 1, file);

	m_CurMesh = mesh;
}

void CMetroModelReader::ReadIndexDataChunk(unsigned int chunkSize, FILE* file)
{
	assert(NULL != m_CurMesh);
	if (!m_CurMesh)
	{
		fseek(file, chunkSize, SEEK_CUR);
		return;
	}

	fread(&m_CurMesh->numIndeces, 4, 1, file);

	m_CurMesh->IBPtr = new MetroModelIndex[m_CurMesh->numIndeces];
	fread(m_CurMesh->IBPtr, m_CurMesh->numIndeces * sizeof(MetroModelIndex), 1, file);

	m_Meshes.push_back(m_CurMesh);

	this->SetMaterials(m_CurMesh);

	m_CurMesh = NULL;
}

void CMetroModelReader::SetMaterials(SMesh* source)
{
	if (NULL != source && NULL != m_LastMaterials && m_LastMaterialsSize > 0)
	{
		char* b  = m_LastMaterials;
		int pos = 0;
		int matIDx = 0;
		while (pos < m_LastMaterialsSize)
		{
			if (*b)
				source->materials[matIDx].append(1, *b);
			else
				matIDx++;

			b++; pos++;
		}
	}

	delete [] m_LastMaterials;
	m_LastMaterials = NULL;
	m_LastMaterialsSize = 0;
}

__forceinline Point3 PointFromDword(DWORD dw)
{
	return Point3(float((dw & 0x00ff0000) >> 16) / 255.0f,
				  float((dw & 0x0000ff00) >>  8) / 255.0f,
				  float((dw & 0x000000ff) >>  0) / 255.0f);
}
__forceinline Point3 FixupPoint(const Point3& pt)
{
	return Point3(pt.z, pt.y, pt.x);
}

int CMetroModelReader::MakeMayaMesh(const SMesh* source, int idx, MObject parent)
{
	const unsigned int numFaces = source->numIndeces / 3;
	unsigned int i;

	MStatus status = MS::kFailure;
//	MFnTransform fnxform;
//	MObject xform = fnxform.create(parent, &status);

	char objName[256] = {0};
	sprintf_s<256>(objName, "object_%02d", idx);

//	fnxform.setName(objName);

	MFloatPointArray vtxs(source->numVerteces);
	for (i = 0; i < source->numVerteces; ++i)
		vtxs.set(i, source->VBPtr[i].pos.z, source->VBPtr[i].pos.y, source->VBPtr[i].pos.x, 1.0f);

	int* indexesAsInt = new int[source->numIndeces];
	if (!indexesAsInt)
		return FALSE;

	for (i = 0; i < source->numIndeces; ++i)
		indexesAsInt[i] = source->IBPtr[i];

	MIntArray fcount(numFaces, 3), facesArray(indexesAsInt, source->numIndeces);

	int count = fcount.length();

	float* u = new float[source->numVerteces];
	float* v = new float[source->numVerteces];

	for (i = 0; i < source->numVerteces; ++i)
	{
		u[i] = source->VBPtr[i].texCoord.x;
		v[i] = 1.0f - source->VBPtr[i].texCoord.y;
	}

	MFloatArray uArray(u, source->numVerteces);
	MFloatArray vArray(v, source->numVerteces);

	MFnMesh fnmesh;
	MObject mesh = fnmesh.create(source->numVerteces, numFaces, vtxs, fcount, facesArray, uArray, vArray, MObject::kNullObj, &status);
	if (MS::kSuccess != status)
	{
		DebugPrint(MString("Could not create mesh from data"));
		return FALSE;
	}

	delete [] indexesAsInt;
	delete [] u;
	delete [] v;

	MString uvSet("map1");
	status = fnmesh.assignUVs(fcount, facesArray, &uvSet);

	status = fnmesh.unlockVertexNormals(facesArray);
	if (MS::kSuccess == status)
	{
		float (*normals)[3] = new float[source->numVerteces][3];
		for(i = 0; i < source->numVerteces; ++i)
		{
			Point3 normal = PointFromDword(source->VBPtr[i].norm).Normalize();
			normals[i][0] = normal.z;
			normals[i][1] = normal.y;
			normals[i][2] = normal.x;
		}

		MVectorArray normsArray(normals, source->numVerteces);
		status = fnmesh.setVertexNormals(normsArray, facesArray, MSpace::kWorld);

		delete [] normals;
	}
	else
	{
		DebugPrint(MString("Could not set normals"));
	}

	this->SetupMayaMesh(source, mesh);

	return TRUE;
}

int CMetroModelReader::SetupMayaMesh(const SMesh* source, MObject& node)
{
	MStatus status;

	std::string texFullPath = this->MakeTexturePath(source->materials[0]);

	MObject texNode = this->FindTextureFileNode(texFullPath);
	if (texNode == MObject::kNullObj)
		texNode = this->CreateTextureFileNode(texFullPath);

	MFnDependencyNode fntex(texNode, &status);
	if (MS::kSuccess != status)
	{
		DebugPrint(MString("Could not create texture node"));
		return FALSE;
	}

	MFnDependencyNode fnshd(CreateShaderNode(MFn::kLambert), &status);
	if (MS::kSuccess != status)
	{
		DebugPrint(MString("Could not create shader node"));
		return FALSE;
	}

	MDGModifier mod;
	ConnectAttr(mod, fntex, "outColor", fnshd, "color");
	//ConnectAttr(mod, fntex, "outTransparency", fnshd, "transparency");
	mod.doIt();

	MObject mat = CreateShadingGroupNode(fnshd);

	AttachShadingGroup(mat, node);

/*	MFnDependencyNode fnsg(mat, &status);
	MFnDagNode fngp(node, &status);

	if (MS::kSuccess == status)
	{
		MString cmd = "sets -e -forceElement " + fnsg.name() + " " + fngp.fullPathName() + ";",
				result;
		MGlobal::executeCommand(cmd, result, false, false);
	}*/

	return TRUE;
}

std::string CMetroModelReader::MakeTexturePath(const std::string& nameFromModel)
{
	if (m_ContentPath.empty())
		return nameFromModel;

	std::string temp = m_ContentPath + "textures\\" + nameFromModel;
	std::string fullPath = temp + ".2048.dds";

	if (!PathFileExists(fullPath.c_str()))
	{
		fullPath = temp + ".1024.dds";
		if (!PathFileExists(fullPath.c_str()))
			fullPath = temp + ".512.dds";
	}

	DebugPrint(MString("Found texture ") + MString(fullPath.c_str()));
	return fullPath;
}



// maya routines
MObject CMetroModelReader::CreatePlace2DNode(const MFnDependencyNode& texFn)
{
	MStatus status;

	MFnDependencyNode p2dFn;
	MObject res = p2dFn.create("place2dTexture", &status );
	if (MS::kSuccess != status)
	{
		DebugPrint(MString("Could not create place2dTexture node"));
		return MObject::kNullObj;
	}

	static const char* srcAttr[] = {"outUV",  "outUvFilterSize","vertexUvOne","vertexUvTwo","vertexUvThree","vertexCameraOne","offset","stagger","coverage","translateFrame","mirrorV","mirrorU","wrapU","wrapV","noiseUV","rotateUV","repeatUV"};
	static const char* dstAttr[] = {"uvCoord","uvFilterSize",   "vertexUvOne","vertexUvTwo","vertexUvThree","vertexCameraOne","offset","stagger","coverage","translateFrame","mirrorV","mirrorU","wrapU","wrapV","noiseUV","rotateUV","repeatUV"};

	MDGModifier mod;
	for (size_t i = 0; i < (sizeof(srcAttr)/sizeof(char *)); ++i)
		ConnectAttr(mod, p2dFn, srcAttr[i], texFn, dstAttr[i]);
	mod.doIt();
	return res;
}

MObject CMetroModelReader::CreateTextureFileNode(const std::string& texName)
{
	MStatus status;

	MObject tobj = this->FindTextureFileNode(texName);
	if (tobj != MObject::kNullObj)
		return tobj;

	MString cmd("shadingNode -asTexture file"), result;
	if (!MGlobal::executeCommand(cmd, result, false, false))
	{
		DebugPrint(MString("Could not executeCommand shadingNode (-asTexture)"));
		return MObject::kNullObj;
	}

	MFnDependencyNode fndep(FindMObject(result), &status);
	if (MS::kSuccess != status)
	{
		DebugPrint(MString("Could not find our tNode"));
		return MObject::kNullObj;
	}

	if (!SetAttribute(fndep, "fileTextureName", texName.c_str()))
		return MObject::kNullObj;

	this->CreatePlace2DNode(fndep.object());
	m_TextureNodes[texName] = fndep.object();

	return fndep.object();
}

const MObject& CMetroModelReader::FindTextureFileNode(const std::string& texName)
{
	texnodesMap::iterator it = m_TextureNodes.find(texName);
	if (it == m_TextureNodes.end())
		return MObject::kNullObj;

	return it->second;
}

MObject CMetroModelReader::CreateShaderNode(MFn::Type type)
{
	MString cmd("shadingNode -asShader "), result;
	switch (type)
	{
		case MFn::kLambert:
			cmd += "lambert "; break;
		case MFn::kPhong:
			cmd += "phong "; break;
		case MFn::kBlinn:
			cmd += "blinn "; break;
		case MFn::kSurfaceShader:
			cmd += "surfaceShader"; break;

		default:
			return MObject::kNullObj;
	}
	if (!MGlobal::executeCommand(cmd, result, false, false))
		return MObject::kNullObj;

	return FindMObject(result);
}

MObject CMetroModelReader::CreateShadingGroupNode(const MFnDependencyNode& fndep)
{
	MStatus status;

	MString result, cmd;
	cmd  = "sets -renderable true -noSurfaceShader true -empty;";
	if (!MGlobal::executeCommand(cmd, result, false, false))
		return MObject::kNullObj;
	
	MFnDependencyNode fnsg(FindMObject(result), &status);
	if (MS::kSuccess != status)
		return MObject::kNullObj;

	MDGModifier mod;
	ConnectAttr(mod, fndep, "outColor", fnsg, "surfaceShader");
	mod.doIt();

	return FindMObject(result);
}
