#ifndef __MetroModelReader_h__
#define __MetroModelReader_h__

#include <string>
#include <vector>
#include <map>
#include <cmath>
#include <algorithm>
#include <cassert>

#include <maya/MObject.h>
#include <maya/MGlobal.h>
#include <maya/MFnTransform.h>
#include <maya/MFnMesh.h>
#include <maya/MPlug.h>
#include <maya/MSelectionList.h>

#include <maya/MIntArray.h>
#include <maya/MFloatArray.h>
#include <maya/MFloatPointArray.h>
#include <maya/MVectorArray.h>
#include <maya/MStringArray.h>


#define ENABLE_PLUGIN_DEBUG 1

#if	ENABLE_PLUGIN_DEBUG
static void _int_DebugPrint(const MString& txt, const char* function)
{
	MGlobal::displayInfo(MString("Metro import: ") + txt + MString(" at func ") + MString(function));
}

#define DebugPrint(debString)	_int_DebugPrint(MString(debString), __FUNCTION__)
#else
#define DebugPrint(debString)	((void)0)
#endif	// ENABLE_PLUGIN_DEBUG



#define METRO_CHUNK_UNKNOWN_1		0x00000001	// iOrange - ????
#define METRO_CHUNK_MATERIALS		0x00000002	// iOrange - some materials names
#define METRO_CHUNK_VERTEXDATA		0x00000003	// iOrange - vertex data chunk
#define METRO_CHUNK_INDEXDATA		0x00000004	// iOrange - indexes data chunk

#define METRO_CHUNK_MESHESHEAP		0x00000009	// iOrange - meshes heap

#define NUM_MESH_MATERIALS			3

#ifndef	DWORD
#	define	DWORD	unsigned int
#endif

struct Point2
{
	float x, y;

	Point2() : x(0.0f), y(0.0f) {}
	Point2(float _x, float _y) : x(_x), y(_y) {}
};

struct Point3
{
	float x, y, z;

	Point3() : x(0.0f), y(0.0f), z(0.0f) {}
	Point3(float _x, float _y, float _z) : x(_x), y(_y), z(_z) {}

	Point3 Normalize(void)
	{
		float len = (float)sqrt(x*x + y*y + z*z);
		return Point3(x / len, y / len, z / len);
	}
};

struct MetroModelVertex
{
	Point3	pos;
	DWORD	a, norm, c;
	Point2	texCoord;
};

typedef unsigned short MetroModelIndex;

struct SMesh
{
	unsigned int		numVerteces;
	unsigned int		numIndeces;

	MetroModelVertex *	VBPtr;
	MetroModelIndex *	IBPtr;

	std::string			materials[NUM_MESH_MATERIALS];
};

class CMetroModelReader
{
public:
	CMetroModelReader();
	~CMetroModelReader();

	int OpenModelFile(const std::string& fileName);
	int ImportToMaya(MObject parent);

private:

	void ReadChunk(unsigned int chunkID, unsigned int chunkSize, FILE* file);
	void ReadMeshesHeapChunk(unsigned int chunkID, unsigned int chunkSize, FILE* file);

	void ReadVertexDataChunk(unsigned int chunkSize, FILE* file);
	void ReadIndexDataChunk(unsigned int chunkSize, FILE* file);
	void SetMaterials(SMesh* source);
	int  MakeMayaMesh(const SMesh* source, int idx, MObject parent);
	int  SetupMayaMesh(const SMesh* source, MObject& node);
	std::string MakeTexturePath(const std::string& nameFromModel);


	// maya routines
	MObject CreatePlace2DNode(const MFnDependencyNode& texFn);
	MObject CreateTextureFileNode(const std::string& texName);
	const MObject& FindTextureFileNode(const std::string& texName);
	MObject CreateShaderNode(MFn::Type type);
	MObject CreateShadingGroupNode(const MFnDependencyNode& fndep);



	typedef std::vector<SMesh *>	meshesVec;
	typedef std::map<std::string, MObject>	texnodesMap;

	meshesVec		m_Meshes;
	SMesh *			m_CurMesh;
	char*			m_LastMaterials;
	int				m_LastMaterialsSize;
	std::string		m_ContentPath;

	MObject			m_InitialShadingGroup;
	texnodesMap		m_TextureNodes;
};

#endif	// __MetroModelReader_h__
